/** @file   LightBeam.h
 * @brief   Declaration of LightBeam class.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/06/17 21:44:40 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_LIGHTBEAM_H
#define H_WWW_LIGHTBEAM_H

#include <string>
#include <vector>
#include "eng2d.h"
#include "MMapAccessInterface.h"

// Forward declarations
struct BITMAP;

namespace WeWantWar {

// Forward declarations
class LightBeamCell;


/** @class  LightBeam
 * @brief   Represents the light beam
 * @author  Tomi Lamminsaari
 */    
class LightBeam : public MMapAccessInterface
{
public:
    /** Constructor.
     * @param   aCellSize       Size of the light cells.
     */
    LightBeam( int aCellSize );
    
    /** Destructor.
     */
    virtual ~LightBeam();
private:
    LightBeam( const LightBeam& aLightBeam );
    LightBeam& operator = ( const LightBeam& aLightBeam );
public: // methods from base class

  /** From MMapAccessInterface
   */
  virtual bool mapCollision( const eng2d::Vec2D& aPosition, int aLayer=0 ) const;
  
  /** From MMapAccessInterface
   */
  virtual bool bulletCollision( const eng2d::Vec2D& aPosition ) const;
  
public: // new methods
    /** Constructs the light beam object from given bitmap. This method creates
     * a matrix of LightBeamCell objects and the brightness/color of those
     * cells are taken the given bitmap.
     * @param   aBitmap         Pointer to bitmap being used.
     * @return  Nonzero if fails.
     */
    int constructFromBitmap( BITMAP* aBitmap );
    
    /** Sets the position and direction angle of this light beam.
     * @param   aOrigin         position of this light beam.
     * @param   aDirection      Direction angle of the light beam.
     */
    void setBeamPosition( const eng2d::Vec2D& aOrigin, float aDirection );
    
    /** Updates the cells that form this light beam
     */
    virtual void update();
    
    /** Applies this light beam on given bitmap.
     * @param   aCanvas         Pointer to bitmap where the light beam should
     *                          be rendered.
     */
    void applyLightBeam( BITMAP* aCanvas, const eng2d::Vec2D& aPos );
    
    /** Returns the position of this light beam
     * @return  Position of the light beam.
     */
    eng2d::Vec2D beamPosition() const;
    
    /** Returns the direction angle of this light beam.
     * @return  Angle (0...256)
     */
    float beamDirection() const;
    
    /** Returns the size of individual light beam cell.
     * @return  Cell size.
     */
    int beamCellSize() const;
    
    /** Returns the total size of the light beam.
     * @param   aWidth          Width goes here.
     * @param   aHeight         Height goes here.
     */
    void lightBeamSize( int& aWidth, int& aHeight ) const;
    
protected:
    /** Adds the given light beam cell to this light beam.
     * @param   aCell           Pointer to lightbeam cell. Ownership is
     *                          transferred.
     */
    void addLightBeamCell( LightBeamCell* aCell );
    
    /** Renders the light beam masks to background buffers.
     */
    void renderLightBeamToBuffer();
    
private:
    /** Array for the light beam cells. */
    std::vector<LightBeamCell*> iLightBeamCells;
    
    /** Position of this light beam. */
    eng2d::Vec2D iPosition;
    
    /** Direction of this light beam. */
    float iDirection;
    
    /** Size of the light beam cells. */
    int iCellSize;
    
    /** Buffer1 */
    BITMAP* iBackBuffer1;
    
    /** Buffer 2. */
    BITMAP* iBackBuffer2;
};


};  // end of namespace

#endif
